// main.cpp
// A complete minimal 3D FPS zombie shooter using SDL2 + OpenGL (GLAD), ENTT ECS, Bullet Physics.
// Modern OpenGL (no immediate mode). Shaders included below as strings.
// External libraries required: SDL2, GLAD, OpenGL, ENTT, Bullet (btBulletDynamicsCommon).
// Assets (models/textures/sfx) provided as download links below in comments.
// Note: This is a single-file compact project intended for demonstration and direct compilation.
// Author: Copilot-style example

// SDL2/ added by EDU
#include <SDL2/SDL.h>
#include <SDL2/SDL_audio.h>
#include <glad/glad.h>
#include <GL/gl.h>

#include <entt/entt.hpp>

#include <btBulletDynamicsCommon.h>

#include <iostream>
#include <vector>
#include <string>
#include <chrono>
#include <random>
#include <cmath>
#include <unordered_map>
#include <memory>
#include <fstream>
#include <sstream>
// added by EDU
#include <stack>

// -------------------- Configuration --------------------
const int WINDOW_WIDTH = 1280;
const int WINDOW_HEIGHT = 720;
const float FOV_DEG = 70.0f;
const float NEAR_PLANE = 0.1f;
const float FAR_PLANE = 1000.0f;

const float PLAYER_HEIGHT = 1.8f;
const float PLAYER_SPEED = 4.5f; // m/s
const float PLAYER_RUN_MULT = 1.6f;
const float MOUSE_SENSITIVITY = 0.0025f;

const int MAX_ZOMBIES_START = 5;
const float WAVE_INTERVAL_SECONDS = 20.0f;

// -------------------- Asset Links --------------------
/*
Models/textures/sfx (small, permissive/public domain or CC0 recommended):
- Cube model used is generated procedurally (no external model).
- Simple crate texture (for walls/floors): https://opengameart.org/content/wooden-crate-texture-512x512 (choose PNG)
- Zombie sound effects:
  - zombie_groan.wav: https://freesound.org/people/qubodup/sounds/219472/ (download .wav)
  - zombie_attack.wav: https://freesound.org/people/InspectorJ/sounds/402032/
- Gun sounds:
  - pistol_shot.wav: https://freesound.org/people/Benboncan/sounds/249807/
  - shotgun_shot.wav: https://freesound.org/people/Benboncan/sounds/249806/
  - mg_shot.wav: https://freesound.org/people/Benboncan/sounds/249808/
- Pickup sound:
  - pickup.wav: https://freesound.org/people/InspectorJ/sounds/352680/
Notes: download WAV files and place them in a "assets/sfx/" directory adjacent to the executable.
Place any texture PNGs in "assets/textures/".
*/

// -------------------- Utility math (minimal) --------------------
struct Vec3 { float x,y,z;
Vec3()=default; 
Vec3(float a,float b,float c):x(a),y(b),z(c){}

/**
    Vec3 operator-(const Vec3& other) const {
        return Vec3(x - other.x, y - other.y, z - other.z);
    }

    Vec3 operator+(const Vec3& other) const {
        return Vec3(x + other.x, y + other.y, z + other.z);
    }

    Vec3 operator*(float scalar) const {
        return Vec3(x * scalar, y * scalar, z * scalar);
    }
**/

    Vec3 normalize() const {
        float len = std::sqrt(x*x + y*y + z*z);
        return Vec3(x / len, y / len, z / len);
    }
    
    Vec3 cross(const Vec3& other) const {
        return Vec3(
            y * other.z - z * other.y,
            z * other.x - x * other.z,
            x * other.y - y * other.x
        );
    }
};
struct Vec2 { float x,y; Vec2()=default; Vec2(float a,float b):x(a),y(b){} };

inline Vec3 operator+(const Vec3 &a,const Vec3 &b){ return Vec3(a.x+b.x,a.y+b.y,a.z+b.z); }
inline Vec3 operator-(const Vec3 &a,const Vec3 &b){ return Vec3(a.x-b.x,a.y-b.y,a.z-b.z); }
inline Vec3 operator*(const Vec3 &a,float s){ return Vec3(a.x*s,a.y*s,a.z*s); }
inline float clampf(float v,float a,float b){ if(v<a) return a; if(v>b) return b; return v; }

struct Mat4 { float m[16]; Mat4(){ for(int i=0;i<16;i++) m[i]=0; } static Mat4 identity(){ Mat4 r; r.m[0]=r.m[5]=r.m[10]=r.m[15]=1.0f; return r; } };

// Minimal matrix operations: perspective, lookAt, translate, rotateY, rotateX, multiply
Mat4 multiply(const Mat4 &a,const Mat4 &b){
    Mat4 r;
    for(int i=0;i<4;i++) for(int j=0;j<4;j++){
        r.m[i*4+j]=0;
        for(int k=0;k<4;k++) r.m[i*4+j]+=a.m[i*4+k]*b.m[k*4+j];
    }
    return r;
}

Mat4 perspective(float fovDeg,float aspect,float zn,float zf){
    float f = 1.0f / tanf(fovDeg * 0.5f * 3.14159265f / 180.0f);
    Mat4 r;
    r.m[0]=f/aspect;
    r.m[5]=f; 
    r.m[10]=(zf+zn)/(zn-zf); 
    r.m[11]=-1.0f; 
    r.m[14]=(2*zf*zn)/(zn-zf);
    return r;
}

Mat4 translate(const Vec3 &t){
    Mat4 r = Mat4::identity();
    r.m[12]=t.x; r.m[13]=t.y; r.m[14]=t.z;
    return r;
}

Mat4 rotateY(float a){
    Mat4 r = Mat4::identity();
    float c=cosf(a), s=sinf(a);
    r.m[0]=c; r.m[2]=s; r.m[8]=-s; r.m[10]=c;
    return r;
}

Mat4 rotateX(float a){
    Mat4 r = Mat4::identity();
    float c=cosf(a), s=sinf(a);
    r.m[5]=c; r.m[6]=-s; r.m[9]=s; r.m[10]=c;
    return r;
}

Mat4 lookAt(const Vec3 &eye,const Vec3 &center,const Vec3 &up){
    Vec3 f = Vec3(center.x-eye.x, center.y-eye.y, center.z-eye.z);
    float fl = sqrtf(f.x*f.x+f.y*f.y+f.z*f.z);
    f = Vec3(f.x/fl, f.y/fl, f.z/fl);
    Vec3 u = up;
    float ul = sqrtf(u.x*u.x+u.y*u.y+u.z*u.z);
    u = Vec3(u.x/ul,u.y/ul,u.z/ul);
    Vec3 s = Vec3(f.y*u.z - f.z*u.y, f.z*u.x - f.x*u.z, f.x*u.y - f.y*u.x);
    float sl = sqrtf(s.x*s.x+s.y*s.y+s.z*s.z);
    s = Vec3(s.x/sl, s.y/sl, s.z/sl);
    Vec3 uu = Vec3(s.y*f.z - s.z*f.y, s.z*f.x - s.x*f.z, s.x*f.y - s.y*f.x);
    Mat4 r = Mat4::identity();
    r.m[0]=s.x; r.m[4]=s.y; r.m[8]=s.z;
    r.m[1]=uu.x; r.m[5]=uu.y; r.m[9]=uu.z;
    r.m[2]=-f.x; r.m[6]=-f.y; r.m[10]=-f.z;
    r.m[12] = -(s.x*eye.x + s.y*eye.y + s.z*eye.z);
    r.m[13] = -(uu.x*eye.x + uu.y*eye.y + uu.z*eye.z);
    r.m[14] =  (f.x*eye.x + f.y*eye.y + f.z*eye.z);
    return r;
}

// -------------------- Shaders --------------------
const char* vertexShaderSrc = R"glsl(
#version 330 core
layout(location=0) in vec3 aPos;
layout(location=1) in vec3 aColor;
uniform mat4 uMVP;
out vec3 vColor;
void main(){
    vColor = aColor;
    gl_Position = uMVP * vec4(aPos,1.0);
}
)glsl";

const char* fragmentShaderSrc = R"glsl(
#version 330 core
in vec3 vColor;
out vec4 FragColor;
void main(){
    FragColor = vec4(vColor,1.0);
}
)glsl";

// -------------------- ECS Components --------------------
struct Transform { Vec3 pos; Vec3 rot; Vec3 scale; Transform():pos(0,0,0),rot(0,0,0),scale(1,1,1){} };
struct RenderMesh { GLuint vao=0; GLuint vbo=0; GLsizei vertexCount=0; Vec3 color; }; // simple colored cube
struct PhysicsBody { btRigidBody* body=nullptr; btCollisionShape* shape=nullptr; };
struct Zombie { int hp=20; int type=0; float speed=1.0f; float lastAttack=0.0f; };
struct Player { int hp=100; int currentWeapon=1; int ammoPistol=50; int ammoShotgun=20; int ammoMG=200; };
struct Projectile { int owner=0; int damage=10; float spawnTime=0.0f; };
struct Pickup { enum Type{Ammo,Health,Power} type; int amount; };
struct WaveController { int currentWave=0; float timeSinceLastWave=0.0f; int zombiesToSpawn=0; };

// -------------------- Globals --------------------
SDL_Window* gWindow = nullptr;
SDL_GLContext gGLContext = nullptr;
SDL_AudioDeviceID audioDevice = 0;
entt::registry gRegistry;
std::unique_ptr<btBroadphaseInterface> broadphase;
std::unique_ptr<btDefaultCollisionConfiguration> collisionConfiguration;
std::unique_ptr<btCollisionDispatcher> dispatcher;
std::unique_ptr<btSequentialImpulseConstraintSolver> solver;
std::unique_ptr<btDiscreteDynamicsWorld> dynamicsWorld;
std::unordered_map<std::string, Uint8*> soundData;
std::unordered_map<std::string, Uint32> soundLen;
SDL_AudioSpec audioSpecWav;

// Simple RNG
std::mt19937 rng((unsigned)std::chrono::high_resolution_clock::now().time_since_epoch().count());

// -------------------- Audio helpers --------------------
bool loadWavToMemory(const std::string &path, Uint8** buffer, Uint32* len, SDL_AudioSpec* spec){
    if(SDL_LoadWAV(path.c_str(), spec, buffer, len)==NULL) return false;
    return true;
}
void playWavOnce(const std::string &name){
    auto itD = soundData.find(name);
    if(itD==soundData.end()) return;
    // Simple one-shot by queueing to audio device
    SDL_ClearQueuedAudio(audioDevice);
    SDL_QueueAudio(audioDevice, itD->second, soundLen[name]);
    SDL_PauseAudioDevice(audioDevice, 0);
}

// -------------------- Rendering helpers (simple cube) --------------------
struct GLProgram { GLuint id=0; GLint loc_mvp=-1; };
GLProgram program;

GLuint makeColoredCubeVAO(Vec3 color){
    // 36 vertices (6 faces * 2 tris * 3 verts)
    float size = 0.2f;
    float vertices[] = {
        // positions           // colors
        -size,-size,-size,    color.x,color.y,color.z,
         size,-size,-size,    color.x,color.y,color.z,
         size, size,-size,    color.x,color.y,color.z,
         size, size,-size,    color.x,color.y,color.z,
        -size, size,-size,    color.x,color.y,color.z,
        -size,-size,-size,    color.x,color.y,color.z,

        -size,-size, size,    color.x,color.y,color.z,
         size,-size, size,    color.x,color.y,color.z,
         size, size, size,    color.x,color.y,color.z,
         size, size, size,    color.x,color.y,color.z,
        -size, size, size,    color.x,color.y,color.z,
        -size,-size, size,    color.x,color.y,color.z,

        -size, size, size,    color.x,color.y,color.z,
        -size, size,-size,    color.x,color.y,color.z,
        -size,-size,-size,    color.x,color.y,color.z,
        -size,-size,-size,    color.x,color.y,color.z,
        -size,-size, size,    color.x,color.y,color.z,
        -size, size, size,    color.x,color.y,color.z,

         size, size, size,    color.x,color.y,color.z,
         size, size,-size,    color.x,color.y,color.z,
         size,-size,-size,    color.x,color.y,color.z,
         size,-size,-size,    color.x,color.y,color.z,
         size,-size, size,    color.x,color.y,color.z,
         size, size, size,    color.x,color.y,color.z,

        -size,-size,-size,    color.x,color.y,color.z,
         size,-size,-size,    color.x,color.y,color.z,
         size,-size, size,    color.x,color.y,color.z,
         size,-size, size,    color.x,color.y,color.z,
        -size,-size, size,    color.x,color.y,color.z,
        -size,-size,-size,    color.x,color.y,color.z,

        -size, size,-size,    color.x,color.y,color.z,
         size, size,-size,    color.x,color.y,color.z,
         size, size, size,    color.x,color.y,color.z,
         size, size, size,    color.x,color.y,color.z,
        -size, size, size,    color.x,color.y,color.z,
        -size, size,-size,    color.x,color.y,color.z
    };
    GLuint vao, vbo;
    glGenVertexArrays(1,&vao);
    glGenBuffers(1,&vbo);
    glBindVertexArray(vao);
    glBindBuffer(GL_ARRAY_BUFFER,vbo);
    glBufferData(GL_ARRAY_BUFFER,sizeof(vertices),vertices,GL_STATIC_DRAW);
    glEnableVertexAttribArray(0); 
    glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,6*sizeof(float),(void*)0);
    glEnableVertexAttribArray(1); 
    glVertexAttribPointer(1,3,GL_FLOAT,GL_FALSE,6*sizeof(float),(void*)(3*sizeof(float)));
    glBindBuffer(GL_ARRAY_BUFFER,0);
    glBindVertexArray(0);
    return vao | (vbo<<16); // pack vao and vbo into single GLuint-ish (we'll unpack)
}
void unpackVAO(GLuint packed, GLuint &vao, GLuint &vbo){
    vao = packed & 0xFFFF;
    vbo = (packed>>16) & 0xFFFF;
}
// But better: return pair via struct; for compactness we used pack, but will not rely on high numbers.

// -------------------- Shader compile --------------------
GLuint compileShader(GLenum type,const char* src){
    GLuint s = glCreateShader(type);
    glShaderSource(s,1,&src,nullptr);
    glCompileShader(s);
    GLint ok; glGetShaderiv(s,GL_COMPILE_STATUS,&ok);
    if(!ok){ char buf[1024]; glGetShaderInfoLog(s,1024,NULL,buf); std::cerr<<"Shader compile error: "<<buf<<"\n"; }
    return s;
}
bool initGLProgram(){
    GLuint vs = compileShader(GL_VERTEX_SHADER, vertexShaderSrc);
    GLuint fs = compileShader(GL_FRAGMENT_SHADER, fragmentShaderSrc);
    program.id = glCreateProgram();
    glAttachShader(program.id, vs);
    glAttachShader(program.id, fs);
    glLinkProgram(program.id);
    GLint ok; glGetProgramiv(program.id,GL_LINK_STATUS,&ok);
    if(!ok){ 
    char buf[1024];
    glGetProgramInfoLog(program.id,1024,NULL,buf); std::cerr<<"Program link error: "<<buf<<"\n"; return false; 
    }
    program.loc_mvp = glGetUniformLocation(program.id,"uMVP");
    glDeleteShader(vs); glDeleteShader(fs);
    return true;
}

// -------------------- Physics helpers --------------------
btRigidBody* createRigidBody(float mass, const btTransform &startTransform, btCollisionShape* shape){
    btVector3 localInertia(0,0,0);
    if(mass!=0.0f) shape->calculateLocalInertia(mass, localInertia);
    btDefaultMotionState* motionState = new btDefaultMotionState(startTransform);
    btRigidBody::btRigidBodyConstructionInfo rbInfo(mass, motionState, shape, localInertia);
    btRigidBody* body = new btRigidBody(rbInfo);
    dynamicsWorld->addRigidBody(body);
    return body;
}

// -------------------- Map generation (maze-like) --------------------
struct MapCell { bool wallNorth=true, wallSouth=true, wallEast=true, wallWest=true; bool visited=false; };
int MAP_W = 25;
int MAP_H = 25;
std::vector<MapCell> mapCells;

inline int cellIndex(int x,int y){ return y*MAP_W + x; }

void generateMaze(){
    mapCells.assign(MAP_W*MAP_H, MapCell());
    std::stack<std::pair<int,int>> stk;
    int sx=1, sy=1;
    mapCells[cellIndex(sx,sy)].visited=true;
    stk.push({sx,sy});
    std::uniform_int_distribution<int> dird(0,3);
    while(!stk.empty()){
        auto [cx,cy] = stk.top();
        std::vector<int> dirs = {0,1,2,3};
        std::shuffle(dirs.begin(), dirs.end(), rng);
        bool moved=false;
        for(int d:dirs){
            int nx=cx, ny=cy;
            if(d==0) ny-=2;
            if(d==1) ny+=2;
            if(d==2) nx+=2;
            if(d==3) nx-=2;
            if(nx>0 && nx<MAP_W-1 && ny>0 && ny<MAP_H-1 && !mapCells[cellIndex(nx,ny)].visited){
                // carve
                mapCells[cellIndex(nx,ny)].visited=true;
                if(d==0){ mapCells[cellIndex(cx,cy)].wallNorth=false; mapCells[cellIndex(cx,cy-1)].wallSouth=false; mapCells[cellIndex(nx,ny)].wallSouth=false; }
                if(d==1){ mapCells[cellIndex(cx,cy)].wallSouth=false; mapCells[cellIndex(cx,cy+1)].wallNorth=false; mapCells[cellIndex(nx,ny)].wallNorth=false; }
                if(d==2){ mapCells[cellIndex(cx,cy)].wallEast=false; mapCells[cellIndex(cx+1,cy)].wallWest=false; mapCells[cellIndex(nx,ny)].wallWest=false; }
                if(d==3){ mapCells[cellIndex(cx,cy)].wallWest=false; mapCells[cellIndex(cx-1,cy)].wallEast=false; mapCells[cellIndex(nx,ny)].wallEast=false; }
                stk.push({nx,ny});
                moved=true;
                break;
            }
        }
        if(!moved) stk.pop();
    }
}

// -------------------- Setup world, entities --------------------
entt::entity playerEntity;
WaveController globalWave;

void spawnWallBlock(const Vec3 &pos, const Vec3 &scale, Vec3 color){
    auto e = gRegistry.create();
    gRegistry.emplace<Transform>(e).pos = pos, gRegistry.get<Transform>(e).scale = scale;
    RenderMesh rm; rm.vao = makeColoredCubeVAO(color); rm.vertexCount = 36; rm.color=color;
    gRegistry.emplace<RenderMesh>(e, rm);
    // physics: static box
    btCollisionShape* shape = new btBoxShape(btVector3(scale.x*0.5f, scale.y*0.5f, scale.z*0.5f));
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(pos.x, pos.y, pos.z));
    btRigidBody* body = createRigidBody(0.0f, t, shape);
    auto &pb = gRegistry.emplace<PhysicsBody>(e);
    pb.body = body; pb.shape = shape;
}

void spawnPickup(const Vec3 &pos, Pickup::Type type, int amount){
    auto e = gRegistry.create();
    gRegistry.emplace<Transform>(e).pos = pos;
    RenderMesh rm; rm.vao = makeColoredCubeVAO(type==Pickup::Health?Vec3(0.0f,1.0f,0.0f):Vec3(1.0f,1.0f,0.0f));
    rm.vertexCount = 36; //rm.color = type==Pickup::Health?Vec3(0,1,0):Vec3(1,1,0);
    gRegistry.emplace<RenderMesh>(e, rm);
    gRegistry.emplace<Pickup>(e).type=type; gRegistry.get<Pickup>(e).amount=amount;
    // physics as kinematic sensor
    btCollisionShape* shape = new btBoxShape(btVector3(0.4f,0.4f,0.4f));
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(pos.x,pos.y,pos.z));
    btRigidBody* body = createRigidBody(0.0f, t, shape);
    body->setCollisionFlags(body->getCollisionFlags() | btCollisionObject::CF_NO_CONTACT_RESPONSE);
    auto &pb = gRegistry.emplace<PhysicsBody>(e);
    pb.body = body; pb.shape = shape;
}

entt::entity spawnZombie(const Vec3 &pos, int type){
    auto e = gRegistry.create();
    Transform tr; tr.pos = pos; tr.scale = Vec3(0.8f,1.6f,0.8f);
    gRegistry.emplace<Transform>(e, tr);
    RenderMesh rm; Vec3 col = (type==0?Vec3(0.4f,0.9f,0.4f):(type==1?Vec3(0.9f,0.4f,0.4f):Vec3(0.4f,0.4f,0.9f)));
    rm.vao = makeColoredCubeVAO(col); rm.vertexCount=36; rm.color=col;
    gRegistry.emplace<RenderMesh>(e, rm);
    Zombie z; z.type=type; z.hp = (type==0?30:(type==1?60:40)); z.speed = (type==0?0.9f:(type==1?0.6f:1.1f));
    gRegistry.emplace<Zombie>(e, z);
    // physics body dynamic capsule approximated by box
    btCollisionShape* shape = new btBoxShape(btVector3(0.4f,0.8f,0.4f));
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(pos.x,pos.y,pos.z));
    btRigidBody* body = createRigidBody(1.0f, t, shape);
    body->setActivationState(DISABLE_DEACTIVATION);
    auto &pb = gRegistry.emplace<PhysicsBody>(e);
    pb.body = body; pb.shape = shape;
    return e;
}

entt::entity spawnPlayer(const Vec3 &pos){
    auto e = gRegistry.create();
    Transform tr; tr.pos = pos; tr.scale = Vec3(0.5f,1.8f,0.5f);
    gRegistry.emplace<Transform>(e,tr);
    gRegistry.emplace<Player>(e);
    // physics capsule handled as kinematic using rigidbody mass 0 and manual movement
    btCollisionShape* shape = new btCapsuleShape(0.3f,1.4f);
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(pos.x,pos.y,pos.z));
    btRigidBody* body = createRigidBody(0.0f, t, shape);
    body->setCollisionFlags(body->getCollisionFlags() | btCollisionObject::CF_KINEMATIC_OBJECT);
    auto &pb = gRegistry.emplace<PhysicsBody>(e);
    pb.body = body; pb.shape = shape;
    return e;
}

entt::entity spawnProjectile(const Vec3 &pos, const Vec3 &dir, int damage){
    auto e = gRegistry.create();
    Transform tr; tr.pos = pos; tr.scale = Vec3(0.1f,0.1f,0.1f);
    gRegistry.emplace<Transform>(e,tr);
    RenderMesh rm; rm.vao = makeColoredCubeVAO(Vec3(1,0.8f,0.1f)); rm.vertexCount=36; rm.color=Vec3(1,0.8f,0.1f);
    gRegistry.emplace<RenderMesh>(e, rm);
    Projectile p; p.damage=damage; p.spawnTime = SDL_GetTicks()/1000.0f;
    gRegistry.emplace<Projectile>(e,p);
    // physics dynamic small sphere approximated by box
    btCollisionShape* shape = new btSphereShape(0.12f);
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(pos.x,pos.y,pos.z));
    btRigidBody* body = createRigidBody(0.2f, t, shape);
    body->setLinearVelocity(btVector3(dir.x*40.0f, dir.y*40.0f, dir.z*40.0f));
    body->setCcdMotionThreshold(0.1f);
    body->setCcdSweptSphereRadius(0.1f);
    auto &pb = gRegistry.emplace<PhysicsBody>(e);
    pb.body = body; pb.shape = shape;
    return e;
}

// -------------------- Game initialization --------------------
bool initSDLAndGL(){
    if(SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO|SDL_INIT_TIMER)!=0){ std::cerr<<"SDL init failed: "<<SDL_GetError()<<"\n"; return false; }
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION,3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION,3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_PROFILE_MASK, SDL_GL_CONTEXT_PROFILE_CORE);
    gWindow = SDL_CreateWindow("Maze Zombie FPS", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, WINDOW_WIDTH, WINDOW_HEIGHT, SDL_WINDOW_OPENGL|SDL_WINDOW_RESIZABLE);
    if(!gWindow){ std::cerr<<"Window create failed: "<<SDL_GetError()<<"\n"; return false; }
    gGLContext = SDL_GL_CreateContext(gWindow);
    if(!gladLoadGLLoader((GLADloadproc)SDL_GL_GetProcAddress)){ std::cerr<<"GLAD init failed\n"; return false; }
    glEnable(GL_DEPTH_TEST);
    return true;
}

bool initAudio(){
    if(SDL_InitSubSystem(SDL_INIT_AUDIO)!=0){ std::cerr<<"Audio init failed\n"; return false; }
    SDL_AudioSpec want;
    SDL_zero(want);
    want.freq = 44100;
    want.format = AUDIO_F32SYS;
    want.channels = 2;
    want.samples = 4096;
    audioDevice = SDL_OpenAudioDevice(NULL, 0, &want, &audioSpecWav, 0);
    if(audioDevice==0){ std::cerr<<"Failed to open audio: "<<SDL_GetError()<<"\n"; return false; }
    // Load WAV files listed in asset links (user must download them)
    std::vector<std::pair<std::string,std::string>> sounds = {
        {"zombie_groan","assets/sfx/zombie_groan.wav"},
        {"zombie_attack","assets/sfx/zombie_attack.wav"},
        {"pistol_shot","assets/sfx/pistol_shot.wav"},
        {"shotgun_shot","assets/sfx/shotgun_shot.wav"},
        {"mg_shot","assets/sfx/mg_shot.wav"},
        {"pickup","assets/sfx/pickup.wav"}
    };
    for(auto &p: sounds){
        Uint8* buf; Uint32 len; SDL_AudioSpec spec;
        if(loadWavToMemory(p.second, &buf, &len, &spec)){
            soundData[p.first] = buf;
            soundLen[p.first] = len;
        } else {
            std::cerr<<"Warning: sound not found: "<<p.second<<"\n";
        }
    }
    return true;
}

void initPhysics(){
    broadphase.reset(new btDbvtBroadphase());
    collisionConfiguration.reset(new btDefaultCollisionConfiguration());
    dispatcher.reset(new btCollisionDispatcher(collisionConfiguration.get()));
    solver.reset(new btSequentialImpulseConstraintSolver());
    dynamicsWorld.reset(new btDiscreteDynamicsWorld(dispatcher.get(), broadphase.get(), solver.get(), collisionConfiguration.get()));
    dynamicsWorld->setGravity(btVector3(0,-9.8f,0));
}

// -------------------- Game world building --------------------
void buildLevel(){
    generateMaze();
    // floor and walls based on maze cells
    float cellSize = 2.0f;
    /**
    for(int y=0;y<MAP_H;y++){
        for(int x=0;x<MAP_W;x++){
            int idx = cellIndex(x,y);
            if(!mapCells[idx].visited) continue;
            float px = (x - MAP_W/2) * cellSize;
            float pz = (y - MAP_H/2) * cellSize;
            // floor tile
            spawnWallBlock(Vec3(px, -0.6f, pz), Vec3(cellSize, 0.4f, cellSize), Vec3(0.6f,0.6f,0.6f));
            // walls for cell boundaries
            if(mapCells[idx].wallNorth) {
            spawnWallBlock(Vec3(px, 0.8f, pz - cellSize*0.5f), Vec3(cellSize,2.0f,0.2f), Vec3(0.4f,0.4f,0.4f));
            }
            if(mapCells[idx].wallSouth) {
            spawnWallBlock(Vec3(px, 0.8f, pz + cellSize*0.5f), Vec3(cellSize,2.0f,0.2f), Vec3(0.4f,0.4f,0.4f));
            }
            if(mapCells[idx].wallWest) {
            spawnWallBlock(Vec3(px - cellSize*0.5f, 0.8f, pz), Vec3(0.2f,2.0f,cellSize), Vec3(0.4f,0.4f,0.4f));
            }
            if(mapCells[idx].wallEast) {
            spawnWallBlock(Vec3(px + cellSize*0.5f, 0.8f, pz), Vec3(0.2f,2.0f,cellSize), Vec3(0.4f,0.4f,0.4f));
            }
            
            
            
        }
    }**/
    // spawn some pickups and open areas
    spawnPickup(Vec3(5.0f,0.0f,5.0f), Pickup::Health, 25);
    spawnPickup(Vec3(4.0f,0.0f,4.0f), Pickup::Ammo, 30);
}

// -------------------- Input state --------------------
struct InputState {
    float mouseDX=0, mouseDY=0;
    bool forward=false, back=false, left=false, right=false, sprint=false;
    bool shoot=false;
    bool keys[512]={0};
} inputState;

void handleSDLEvent(const SDL_Event &ev){
    if(ev.type==SDL_MOUSEMOTION){
        inputState.mouseDX += ev.motion.xrel;
        inputState.mouseDY += ev.motion.yrel;
    } else if(ev.type==SDL_MOUSEBUTTONDOWN){
        if(ev.button.button==SDL_BUTTON_LEFT) inputState.shoot=true;
    } else if(ev.type==SDL_MOUSEBUTTONUP){
        if(ev.button.button==SDL_BUTTON_LEFT) inputState.shoot=false;
    } else if(ev.type==SDL_KEYDOWN){
        inputState.keys[ev.key.keysym.scancode]=true;
        if(ev.key.keysym.sym==SDLK_w) inputState.forward=true;
        if(ev.key.keysym.sym==SDLK_s) inputState.back=true;
        if(ev.key.keysym.sym==SDLK_a) inputState.left=true;
        if(ev.key.keysym.sym==SDLK_d) inputState.right=true;
        if(ev.key.keysym.sym==SDLK_LSHIFT) inputState.sprint=true;
        if(ev.key.keysym.sym==SDLK_1) gRegistry.get<Player>(playerEntity).currentWeapon=1;
        if(ev.key.keysym.sym==SDLK_2) gRegistry.get<Player>(playerEntity).currentWeapon=2;
        if(ev.key.keysym.sym==SDLK_3) gRegistry.get<Player>(playerEntity).currentWeapon=3;
    } else if(ev.type==SDL_KEYUP){
        inputState.keys[ev.key.keysym.scancode]=false;
        if(ev.key.keysym.sym==SDLK_w) inputState.forward=false;
        if(ev.key.keysym.sym==SDLK_s) inputState.back=false;
        if(ev.key.keysym.sym==SDLK_a) inputState.left=false;
        if(ev.key.keysym.sym==SDLK_d) inputState.right=false;
        if(ev.key.keysym.sym==SDLK_LSHIFT) inputState.sprint=false;
    }
}

// -------------------- Game loop helpers --------------------
float lastFrameTime = 0.0f;
float getTimeSeconds(){ return SDL_GetTicks() / 1000.0f; }

void processPlayerMovement(float dt, float &yaw, float &pitch){
    // read player transform, physics body
    auto &tr = gRegistry.get<Transform>(playerEntity);
    auto &pb = gRegistry.get<PhysicsBody>(playerEntity);
    Vec3 forwardVec = Vec3(sinf(yaw), 0, cosf(yaw));
    Vec3 rightVec = Vec3(cosf(yaw), 0, -sinf(yaw));
    Vec3 vel(0,0,0);
    float speed = PLAYER_SPEED * (inputState.sprint?PLAYER_RUN_MULT:1.0f);
    if(inputState.forward) vel = vel + forwardVec;
    if(inputState.back) vel = vel - forwardVec;
    if(inputState.left) vel = vel - rightVec;
    if(inputState.right) vel = vel + rightVec;
    // normalize
    float len = sqrtf(vel.x*vel.x + vel.y*vel.y + vel.z*vel.z);
    if(len>0.001f){ vel = vel * (speed / len); }
    // apply movement while keeping player above ground
    tr.pos.x += vel.x * dt;
    tr.pos.z += vel.z * dt;
    // clamp inside map bounds
    float mapRadius = (std::max(MAP_W,MAP_H) * 2.0f);
    tr.pos.x = clampf(tr.pos.x, -mapRadius, mapRadius);
    tr.pos.z = clampf(tr.pos.z, -mapRadius, mapRadius);
    // update kinematic physics body
    btTransform t; t.setIdentity(); t.setOrigin(btVector3(tr.pos.x, PLAYER_HEIGHT*0.5f, tr.pos.z));
    pb.body->getMotionState()->setWorldTransform(t);
    pb.body->setWorldTransform(t);
    // mouse look
    yaw -= inputState.mouseDX * MOUSE_SENSITIVITY;
    pitch -= inputState.mouseDY * MOUSE_SENSITIVITY;
    pitch = clampf(pitch, -1.4f, 1.4f);
    inputState.mouseDX = inputState.mouseDY = 0;
}

void spawnWaveIfNeeded(float dt){
    globalWave.timeSinceLastWave += dt;
    if(globalWave.currentWave==0 || globalWave.timeSinceLastWave >= WAVE_INTERVAL_SECONDS){
        globalWave.currentWave++;
        globalWave.timeSinceLastWave = 0.0f;
        globalWave.zombiesToSpawn = MAX_ZOMBIES_START + globalWave.currentWave * 3;
        // spawn zombies at map edges in waves
        std::uniform_real_distribution<float> off(-MAP_W, MAP_W);
        for(int i=0;i<globalWave.zombiesToSpawn;i++){
            float ex = (off(rng)) * 1.9f;
            float ez = (off(rng)) * 1.9f;
            int type = i%3;
            spawnZombie(Vec3(ex,0.0f,ez), type);
        }
        // reward player for previous wave (if any)
        playWavOnce("pickup");
        auto &player = gRegistry.get<Player>(playerEntity);
        player.ammoPistol += 10 * globalWave.currentWave;
        player.ammoShotgun += 5 * globalWave.currentWave;
        player.ammoMG += 30 * globalWave.currentWave;
    }
}

// -------------------- Gameplay: zombie AI, projectiles collisions --------------------
void updateZombies(float dt){
    // get player pos
    auto &ptr = gRegistry.get<Transform>(playerEntity);
    Vec3 ppos = ptr.pos;
    float now = getTimeSeconds();
    gRegistry.view<Zombie,Transform,PhysicsBody>().each([&](auto &z, auto &t, auto &pb){
        // simple move toward player
        btTransform bttr; pb.body->getMotionState()->getWorldTransform(bttr);
        btVector3 bpos = bttr.getOrigin();
        Vec3 zpos(bpos.x(), bpos.y(), bpos.z());
        Vec3 dir = Vec3(ppos.x - zpos.x, 0, ppos.z - zpos.z);
        float dlen = sqrtf(dir.x*dir.x + dir.z*dir.z);
        if(dlen>0.01f){
            dir.x /= dlen; dir.z /= dlen;
            // special mechanics by type
            if(z.type==0){
                // slow but piercing: moves slowly, strong attack when close
                pb.body->setLinearVelocity(btVector3(dir.x * z.speed * 1.0f, 0, dir.z * z.speed * 1.0f));
                if(dlen < 1.5f && now - z.lastAttack > 1.5f){
                    // melee attack
                    gRegistry.get<Player>(playerEntity).hp -= 12;
                    playWavOnce("zombie_attack");
                    z.lastAttack = now;
                }
            } else if(z.type==1){
                // tank: very slow but high HP, pushes player
                pb.body->setLinearVelocity(btVector3(dir.x * z.speed * 0.6f, 0, dir.z * z.speed * 0.6f));
                if(dlen < 1.8f && now - z.lastAttack > 2.5f){
                    gRegistry.get<Player>(playerEntity).hp -= 18;
                    playWavOnce("zombie_attack");
                    z.lastAttack = now;
                }
            } else {
                // runner: faster and can dash
                float sp = z.speed * ( (dlen<5.0f && now - z.lastAttack > 4.0f) ? 3.0f : 1.0f );
                pb.body->setLinearVelocity(btVector3(dir.x * sp, 0, dir.z * sp));
                if(dlen < 1.2f && now - z.lastAttack > 1.0f){
                    gRegistry.get<Player>(playerEntity).hp -= 10;
                    playWavOnce("zombie_attack");
                    z.lastAttack = now;
                }
            }
        } else {
            pb.body->setLinearVelocity(btVector3(0,0,0));
        }
        // sync transform to physics
        btTransform w; pb.body->getMotionState()->getWorldTransform(w);
        btVector3 p = w.getOrigin();
        t.pos = Vec3(p.x(), p.y(), p.z());
    });
}

void processProjectileCollisions(){
    int numManifolds = dynamicsWorld->getDispatcher()->getNumManifolds();
    for(int i=0;i<numManifolds;i++){
        btPersistentManifold* contactManifold = dynamicsWorld->getDispatcher()->getManifoldByIndexInternal(i);
        const btCollisionObject* obA = contactManifold->getBody0();
        const btCollisionObject* obB = contactManifold->getBody1();
        for(int p=0;p<contactManifold->getNumContacts();p++){
            btManifoldPoint& pt = contactManifold->getContactPoint(p);
            if(pt.getDistance() < 0.0f){
                // Try to find projectile and zombie entities whose PhysicsBody::body matches obA or obB
                entt::entity eProj = entt::null;
                entt::entity eZombie = entt::null;
                gRegistry.view<PhysicsBody,Projectile>().each([&](entt::entity e, PhysicsBody &pb, Projectile &proj){
                    if(pb.body==obA || pb.body==obB) eProj = e;
                });
                gRegistry.view<PhysicsBody,Zombie>().each([&](entt::entity e, PhysicsBody &pb, Zombie &z){
                    if(pb.body==obA || pb.body==obB) eZombie = e;
                });
                if(eProj!=entt::null && eZombie!=entt::null){
                    Projectile &pr = gRegistry.get<Projectile>(eProj);
                    Zombie &z = gRegistry.get<Zombie>(eZombie);
                    z.hp -= pr.damage;
                    playWavOnce("zombie_groan");
                    // destroy projectile
                    auto &ppb = gRegistry.get<PhysicsBody>(eProj);
                    dynamicsWorld->removeRigidBody(ppb.body);
                    delete ppb.body->getMotionState(); delete ppb.body; delete ppb.shape;
                    gRegistry.destroy(eProj);
                    if(z.hp<=0){
                        // drop rewards
                        spawnPickup(gRegistry.get<Transform>(eZombie).pos, Pickup::Ammo, 10);
                        spawnPickup(gRegistry.get<Transform>(eZombie).pos + Vec3(1,0,0), Pickup::Health, 10);
                        // remove zombie physics
                        auto &zpb = gRegistry.get<PhysicsBody>(eZombie);
                        dynamicsWorld->removeRigidBody(zpb.body);
                        delete zpb.body->getMotionState(); delete zpb.body; delete zpb.shape;
                        gRegistry.destroy(eZombie);
                    }
                }
            }
        }
    }
}

Mat4 myLookAt(Vec3 center, Vec3 eye, Vec3 up) {
   Vec3 f;
   f.x = center.x - eye.x;
   f.y = center.y - eye.y;
   f.z = center.z - eye.z;
   
   f = f.normalize();
   up = up.normalize();
   
   auto s = f.cross(up);
   auto snorm = s.normalize();
   auto u = snorm.cross(f);
   
   Mat4 result;
   result.m[0] = s.x;
   result.m[1] = s.y;
   result.m[2] = s.z;
   
   result.m[5] = u.x;
   result.m[6] = u.y;
   result.m[7] = u.z;
   
   result.m[9] = -1.0f * f.x;
   result.m[10] = -1.0f * f.y;
   result.m[11] = -1.0f * f.z;
   
   return result;
}

// -------------------- Rendering --------------------
void renderScene(float yaw, float pitch){
    glViewport(0,0,WINDOW_WIDTH,WINDOW_HEIGHT);
    glClearColor(0.2f,0.2f,0.25f,1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glUseProgram(program.id);
    // camera
    auto const& ptr = gRegistry.get<Transform>(playerEntity);
    Vec3 camPos = ptr.pos + Vec3(0, PLAYER_HEIGHT*0.4f, 0);
    Vec3 forward = Vec3(sinf(yaw)*cosf(pitch), sinf(pitch), sinf(yaw)*cosf(pitch));
    forward = forward.normalize();
    Vec3 target = camPos + forward;
    Mat4 view = lookAt(camPos, target, Vec3(0,1,0));
    //camPos = Vec3(-4.0f, 0.5f, -4.0f); // hack for now
    target = Vec3(-1.0f, 0.0f, -1.0f); // hack
    view = myLookAt(camPos, target, Vec3(0,1,0));
    Mat4 proj = perspective(FOV_DEG, float(WINDOW_WIDTH)/float(WINDOW_HEIGHT), NEAR_PLANE, FAR_PLANE);
    Mat4 vp = multiply(proj, view);
    // render all entities with RenderMesh + Transform
    gRegistry.view<RenderMesh,Transform>().each([&](auto &rm, auto &tr){
        Mat4 model = translate(tr.pos);
        Mat4 mvp = multiply(vp, model);
        glUniformMatrix4fv(program.loc_mvp, 1, GL_FALSE, mvp.m);
        // unpack VAO/VBO
        GLuint vao = rm.vao & 0xFFFF;
        glBindVertexArray(vao);
        glDrawArrays(GL_TRIANGLES, 0, rm.vertexCount);
    });

    glBindVertexArray(0);
    glUseProgram(0);
    // HUD: draw simple health/ammo text via SDL (overlay)
    // We'll use SDL to draw basic text by drawing colored rectangles for meters (no font dependency)
    // For simplicity, we'll blit a semi-transparent quad via OpenGL orthographic projection.
    // Health bar
    int health = gRegistry.get<Player>(playerEntity).hp;
    if(health<0) health=0;
    float healthFrac = float(health) / 100.0f;
    // Simple overlay quad draw (immediate-ish using GL_TRIANGLES but with modern funcs)
    // Prepare orthographic matrix
    glDisable(GL_DEPTH_TEST);
    // small colored rect shader reuse: use same shader with identity transform -> transform vertices in NDC
    // We'll construct and draw using glBegin alternative - but immediate mode forbidden; so create temporary VBO
    struct QuadVert { float x,y,z; float r,g,b; };
    QuadVert quadVerts[6];
    auto drawRect = [&](float x, float y, float w, float h, Vec3 color){
        float nx = (x / WINDOW_WIDTH)*2.0f -1.0f;
        float ny = -((y / WINDOW_HEIGHT)*2.0f -1.0f);
        float nw = (w / WINDOW_WIDTH)*2.0f;
        float nh = (h / WINDOW_HEIGHT)*2.0f;
        QuadVert v[6] = {
            {nx, ny, 0, color.x,color.y,color.z},
            {nx+nw, ny,0,color.x,color.y,color.z},
            {nx+nw, ny-nh,0,color.x,color.y,color.z},
            {nx+nw, ny-nh,0,color.x,color.y,color.z},
            {nx, ny-nh,0,color.x,color.y,color.z},
            {nx, ny,0,color.x,color.y,color.z}
        };
        GLuint tmpvao, tmpvbo;
        glGenVertexArrays(1,&tmpvao);
        glGenBuffers(1,&tmpvbo);
        glBindVertexArray(tmpvao);
        glBindBuffer(GL_ARRAY_BUFFER,tmpvbo);
        glBufferData(GL_ARRAY_BUFFER,sizeof(v),v,GL_DYNAMIC_DRAW);
        glEnableVertexAttribArray(0); 
        glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,6*sizeof(float),(void*)0);
        glEnableVertexAttribArray(1); 
        glVertexAttribPointer(1,3,GL_FLOAT,GL_FALSE,6*sizeof(float),(void*)(3*sizeof(float)));
        // identity transform
        Mat4 id = Mat4::identity();
        glUniformMatrix4fv(program.loc_mvp, 1, GL_FALSE, id.m);
        glDrawArrays(GL_TRIANGLES,0,6);
        glBindVertexArray(0);
        glDeleteBuffers(1,&tmpvbo);
        glDeleteVertexArrays(1,&tmpvao);
    };
    // health background
    glUseProgram(program.id);
    drawRect(20, 40, 300, 24, Vec3(0.2f,0.2f,0.2f));
    drawRect(20, 40, 300*healthFrac, 24, Vec3(1.0f-healthFrac, healthFrac, 0.0f));
    // ammo display as small bars
    auto &pl = gRegistry.get<Player>(playerEntity);
    int curAmmo = (pl.currentWeapon==1?pl.ammoPistol:(pl.currentWeapon==2?pl.ammoShotgun:pl.ammoMG));
    // ammo bar
    drawRect(20, 80, 150, 18, Vec3(0.2f,0.2f,0.2f));
    float af = clampf(curAmmo/200.0f, 0.0f, 1.0f);
    drawRect(20, 80, 150*af, 18, Vec3(0.2f,0.6f,1.0f));
    glUseProgram(0);
    glEnable(GL_DEPTH_TEST);
}

// -------------------- Main --------------------
int main(int argc,char** argv){
    if(!initSDLAndGL()) return -1;
    if(!initAudio()) std::cerr<<"Audio not fully initialized\n";
    initPhysics();
    if(!initGLProgram()) return -1;
    // create player
    playerEntity = spawnPlayer(Vec3(0,0,0));
    // build level
    buildLevel();
    // spawn initial zombies
    spawnWaveIfNeeded(9999.0f);
    // capture mouse
    SDL_SetRelativeMouseMode(SDL_TRUE);
    bool running=true;
    float yaw = 0.0f, pitch = 0.0f;
    lastFrameTime = getTimeSeconds();
    auto lastLogTime = lastFrameTime;
    // main loop
    while(running){
        float now = getTimeSeconds();
        float dt = now - lastFrameTime;
        if(dt>0.1f) dt = 0.1f;
        lastFrameTime = now;
        SDL_Event ev;
        while(SDL_PollEvent(&ev)){
            if(ev.type==SDL_QUIT) running=false;
            if(ev.type==SDL_KEYDOWN && ev.key.keysym.sym==SDLK_ESCAPE) running=false;
            handleSDLEvent(ev);
        }
        // gameplay
        processPlayerMovement(dt, yaw, pitch);
        spawnWaveIfNeeded(dt);
        updateZombies(dt);
        // process shooting input
        static float lastShotTime = 0.0f;
        lastShotTime += dt;
        auto &pl = gRegistry.get<Player>(playerEntity);
        if(inputState.shoot){
            // weapon behavior
            int cw = pl.currentWeapon;
            if(cw==1 && pl.ammoPistol>0 && lastShotTime>0.2f){
                // pistol single shot
                Vec3 camPos = gRegistry.get<Transform>(playerEntity).pos + Vec3(0,PLAYER_HEIGHT*0.5f,0);
                Vec3 dir = Vec3(sinf(yaw), sinf(pitch), cosf(yaw)); // approximate
                spawnProjectile(camPos + dir*1.0f, dir, 18);
                pl.ammoPistol--;
                lastShotTime=0.0f;
                playWavOnce("pistol_shot");
            } else if(cw==2 && pl.ammoShotgun>0 && lastShotTime>0.8f){
                // shotgun: fires multiple spread projectiles
                Vec3 camPos = gRegistry.get<Transform>(playerEntity).pos + Vec3(0,PLAYER_HEIGHT*0.5f,0);
                for(int i=0;i<8;i++){
                    float rx = ((rng()%100)/100.0f - 0.5f) * 0.2f;
                    float ry = ((rng()%100)/100.0f - 0.5f) * 0.08f;
                    Vec3 dir = Vec3(sinf(yaw)+rx, sinf(pitch)+ry, cosf(yaw)+rx);
                    float len = sqrtf(dir.x*dir.x + dir.y*dir.y + dir.z*dir.z);
                    dir = dir*(1.0f/len);
                     spawnProjectile(camPos + dir*1.0f, dir, 12);
                }
                pl.ammoShotgun -= 1;
                lastShotTime = 0.0f;
                playWavOnce("shotgun_shot");
            } else if(cw==3 && pl.ammoMG>0 && lastShotTime>0.06f){
                Vec3 camPos = gRegistry.get<Transform>(playerEntity).pos + Vec3(0,PLAYER_HEIGHT*0.5f,0);
                Vec3 dir = Vec3(sinf(yaw)+((rng()%100)/100.0f-0.5f)*0.03f, sinf(pitch)+((rng()%100)/100.0f-0.5f)*0.02f, cosf(yaw));
                float len = sqrtf(dir.x*dir.x + dir.y*dir.y + dir.z*dir.z);
                dir = dir*(1.0f/len);
               spawnProjectile(camPos + dir*1.0f, dir, 8);
                pl.ammoMG--;
                lastShotTime=0.0f;
                playWavOnce("mg_shot");
            }
        }
        // step physics
        dynamicsWorld->stepSimulation(dt, 10);
        processProjectileCollisions();
        // pickups: simple proximity check
        auto &pt = gRegistry.get<Transform>(playerEntity);
        if (lastFrameTime - lastLogTime> 2.0) {
          std::cout << "player transform: " << pt.pos.x << " - " << pt.pos.y << " - " << pt.pos.z << std::endl;;
          
          lastLogTime = lastFrameTime;
        }
        gRegistry.view<Pickup,Transform>().each([&](auto &pk, auto &tr){
            Vec3 d = Vec3(tr.pos.x - pt.pos.x, tr.pos.y - pt.pos.y, tr.pos.z - pt.pos.z);
            float dist = sqrtf(d.x*d.x + d.z*d.z);
            if(dist < 1.2f){
                if(pk.type==Pickup::Health){
                    auto &player = gRegistry.get<Player>(playerEntity);
                    player.hp = std::min(100, player.hp + pk.amount);
                } else if(pk.type==Pickup::Ammo){
                    auto &player = gRegistry.get<Player>(playerEntity);
                    player.ammoPistol += pk.amount;
                    player.ammoShotgun += pk.amount/2;
                    player.ammoMG += pk.amount*2;
                }
                playWavOnce("pickup");
                // remove physics and entity
                // find entity by transform: expensive but fine for demo
                // Mark for removal by storing entity to destroy later - for brevity destroy now
                // remove physics
                // NOTE: This lambda has 'tr' by reference from view, safe to get entity via view:complex; Instead we'll collect to remove list
            }
        });
        // remove pickups that were collected: for simplicity, iterate and remove ones close
        std::vector<entt::entity> pickupsToRemove;
        gRegistry.view<Pickup,Transform,PhysicsBody>().each([&](entt::entity e, Pickup &pk, Transform &tr, PhysicsBody &pb){
            Vec3 d = Vec3(tr.pos.x - pt.pos.x, tr.pos.y - pt.pos.y, tr.pos.z - pt.pos.z);
            float dist = sqrtf(d.x*d.x + d.z*d.z);
            if(dist < 1.2f){
                pickupsToRemove.push_back(e);
            }
        });
        for(auto e: pickupsToRemove){
            auto &pb = gRegistry.get<PhysicsBody>(e);
            dynamicsWorld->removeRigidBody(pb.body);
            delete pb.body->getMotionState(); delete pb.body; delete pb.shape;
            gRegistry.destroy(e);
        }
        // check player dead
        if(gRegistry.get<Player>(playerEntity).hp<=0){
            std::cerr<<"Player died. Exiting.\n";
            running=false;
        }
        // render
        renderScene(yaw, pitch);
        SDL_GL_SwapWindow(gWindow);
    }
    // cleanup: free audio
    for(auto &p: soundData) SDL_FreeWAV(p.second);
    // delete physics objects left (simple)
    // ... (omitted for brevity)
    SDL_Quit();
    return 0;
}
