// train_defender_box2d.cpp
// Train Defender (top-down) using SDL2, SDL_ttf, SDL_mixer, ENTT, and Box2D.
// Requirements: SDL2, SDL2_ttf, SDL2_mixer, Box2D, ENTT. Provide BMP + TTF + audio assets next to executable.

#include <SDL2/SDL.h>
#include <SDL2/SDL_ttf.h>
#include <SDL2/SDL_mixer.h>

#include <entt/entt.hpp>
#include <box2d/box2d.h>

#include <vector>
#include <string>
#include <cmath>
#include <random>
#include <chrono>
#include <iostream>
#include <memory>
#include <unordered_map>

static const int WINDOW_W = 1024;
static const int WINDOW_H = 600;

// --- Asset filenames (place these next to the executable) ---
const char* ASSET_TRAIN = "train.bmp";
const char* ASSET_MONSTER = "monster.bmp";
const char* ASSET_TRACK = "track.bmp";
const char* ASSET_PROJECTILE = "projectile.bmp";
const char* ASSET_UI = "ui.bmp";
const char* ASSET_FONT = "/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf"; // EDU change the path
const char* ASSET_BG_MUSIC = "music.ogg";
const char* ASSET_FIRE_SOUND = "fire.wav";
const char* ASSET_HIT_SOUND = "hit.wav";

// --- Game constants ---
const float PIXELS_PER_METER = 32.0f; // box2d <-> screen scaling
const float INV_PIXELS_PER_METER = 1.0f / PIXELS_PER_METER;
const float TRACK_LENGTH = 120.0f; // meters in world space
const float MISSION_TIME_SECONDS = 120.0f;
const int MAX_CARGO_HEALTH = 100;

// --- Helper types ---
struct Texture { SDL_Texture* tex = nullptr; int w = 0, h = 0; };
struct Vec2 { float x; float y; };
static uint32_t now_ms() { return SDL_GetTicks(); }

// --- Globals ---
SDL_Window* gWindow = nullptr;
SDL_Renderer* gRenderer = nullptr;
TTF_Font* gFont = nullptr;
Mix_Music* gMusic = nullptr;
Mix_Chunk* gFireSound = nullptr;
Mix_Chunk* gHitSound = nullptr;

Texture gTrainTex, gMonsterTex, gTrackTex, gProjTex, gUiTex;

entt::registry registry;
std::mt19937 rng((unsigned)std::chrono::high_resolution_clock::now().time_since_epoch().count());

// Box2D world
std::unique_ptr<b2World> worldPtr;

// --- Components ---
struct Transform { Vec2 pos; float rot = 0.0f; };
struct Sprite { Texture* tex; int w,h; };
struct Collider { int radius_px; };
struct Train {
    float progress_m = 0.0f;         // progress along track in meters
    float speed_mps = 2.5f;          // meters per second (target speed)
    float cargo_health = MAX_CARGO_HEALTH;
    double mission_clock = 0.0;
    bool active = true;
    bool boost = false;
};
struct Monster { int hp; };
struct Projectile { int damage; int owner; double life_ms; };
struct Body { b2Body* body = nullptr; }; // Box2D body pointer attached to entity
struct Lifetime { double remain_ms; };

// --- Camera (keeps train centered) ---
struct Camera {
    Vec2 center_world; // world coordinates at screen center (meters)
    SDL_Point world_to_screen(const Vec2 &wpos) const {
        float sx = (wpos.x - center_world.x) * PIXELS_PER_METER + WINDOW_W * 0.5f;
        float sy = (wpos.y - center_world.y) * PIXELS_PER_METER + WINDOW_H * 0.5f;
        return SDL_Point{ (int)std::round(sx), (int)std::round(sy) };
    }
    Vec2 screen_to_world(int sx, int sy) const {
        return Vec2{ (sx - WINDOW_W*0.5f) * INV_PIXELS_PER_METER + center_world.x,
                     (sy - WINDOW_H*0.5f) * INV_PIXELS_PER_METER + center_world.y };
    }
} camera;

// --- Utility functions ---
bool loadTextureBMP(const char* path, Texture &out) {
    SDL_Surface* surf = SDL_LoadBMP(path);
    if (!surf) {
        std::cerr << "Failed to load BMP '" << path << "': " << SDL_GetError() << "\n";
        return false;
    }
    out.tex = SDL_CreateTextureFromSurface(gRenderer, surf);
    out.w = surf->w; out.h = surf->h;
    SDL_FreeSurface(surf);
    return out.tex != nullptr;
}
void draw_text(const std::string &txt, int x, int y, SDL_Color color) {
    if (!gFont) return;
    SDL_Surface* surf = TTF_RenderUTF8_Blended(gFont, txt.c_str(), color);
    if (!surf) return;
    SDL_Texture* t = SDL_CreateTextureFromSurface(gRenderer, surf);
    SDL_Rect dst{ x, y, surf->w, surf->h };
    SDL_FreeSurface(surf);
    SDL_RenderCopy(gRenderer, t, nullptr, &dst);
    SDL_DestroyTexture(t);
}

// --- Track mapping: progress (meters 0..TRACK_LENGTH) -> world coords (meters) ---
// We'll layout the track along X axis in world meters, with a sine offset on Y for interest.
Vec2 track_to_world(float progress_m) {
    float t = progress_m / TRACK_LENGTH; // 0..1
    float wx = progress_m; // x in meters
    float wy = std::sin(t * 8.0f) * 3.5f; // meters vertical wiggle
    return Vec2{ wx, wy };
}

// --- ENTT <-> Box2D contact listener to process collisions ---
class ContactListener : public b2ContactListener {
public:
    void BeginContact(b2Contact* contact) override {
        b2Fixture* a = contact->GetFixtureA();
        b2Fixture* b = contact->GetFixtureB();
        // EDU - refactor here
        entt::entity ea, eb;
        if (a->GetUserData().pointer) {
          auto ptr = static_cast<uintptr_t>(a->GetUserData().pointer);
          ea = static_cast<entt::entity>(ptr);
        }
        if (b->GetUserData().pointer) {
          auto ptr = static_cast<uintptr_t>(b->GetUserData().pointer);
          eb = static_cast<entt::entity>(ptr);
        }


        //void* ad = a->GetUserData().pointer ? reinterpret_cast<void*>(a->GetUserData().pointer) : nullptr;
        //void* bd = b->GetUserData().pointer ? reinterpret_cast<void*>(b->GetUserData().pointer) : nullptr;
        //entt::entity ea = entt::null, eb = entt::null;
        //if (ad) ea = reinterpret_cast<entt::entity>(ad);
        //if (bd) eb = reinterpret_cast<entt::entity>(bd);

        // if projectile hits monster or monster hits train -> handle damage
        handle_collision(ea, eb);
        handle_collision(eb, ea);
    }
private:
    void handle_collision(entt::entity a, entt::entity b) {
        if (a==entt::null || b==entt::null) return;
        if (!registry.valid(a) || !registry.valid(b)) return;

        // projectile -> monster
        if (registry.any_of<Projectile>(a) && registry.any_of<Monster>(b)) {
            auto &proj = registry.get<Projectile>(a);
            auto &mon = registry.get<Monster>(b);
            mon.hp -= proj.damage;
            if (gHitSound) Mix_PlayChannel(-1, gHitSound, 0);
            // destroy projectile entity and possibly monster (delete handled by game loop)
            registry.emplace_or_replace<Lifetime>(a, Lifetime{0.0});
        }

        // monster -> train
        if (registry.any_of<Monster>(a) && registry.any_of<Train>(b)) {
            auto &train = registry.get<Train>(b);
            train.cargo_health -= 10;
            if (gHitSound) Mix_PlayChannel(-1, gHitSound, 0);
            // destroy monster
            registry.emplace_or_replace<Lifetime>(a, Lifetime{0.0});
        }

        // projectile -> world or projectile -> train (friendly fire ignored)
        if (registry.any_of<Projectile>(a) && registry.any_of<Train>(b)) {
            // optionally ignore friendly fire; we do not damage train by player projectiles
            registry.emplace_or_replace<Lifetime>(a, Lifetime{0.0});
        }
    }
};

// --- Spawn helpers that create ENTT entity and Box2D body, attach Body component and set user data ---
entt::entity create_monster(float progress_m) {
    Vec2 pos = track_to_world(progress_m);
    b2BodyDef bd;
    bd.type = b2_dynamicBody;
    bd.position.Set(pos.x, pos.y);
    b2Body* body = worldPtr->CreateBody(&bd);
    b2CircleShape shape;
    shape.m_radius = 0.45f; // ~14 px
    b2FixtureDef fd;
    fd.shape = &shape;
    fd.density = 1.0f;
    fd.friction = 0.3f;
    fd.restitution = 0.0f;
    auto ent = registry.create();
    // set user data pointer to entity (store entity value in fixture user data)
    b2Fixture* fx = body->CreateFixture(&fd);
    // EDU - change to static_cast
    fx->GetUserData().pointer = static_cast<uintptr_t>(ent);
    registry.emplace<Body>(ent, Body{body});
    registry.emplace<Transform>(ent, Transform{ {pos.x, pos.y}, 0.0f });
    registry.emplace<Sprite>(ent, Sprite{ &gMonsterTex, 32, 32 });
    registry.emplace<Monster>(ent, Monster{ std::uniform_int_distribution<int>(20, 45)(rng) });
    registry.emplace<Collider>(ent, Collider{14});
    return ent;
}

entt::entity create_projectile(const Vec2 &world_pos, const Vec2 &dir, int damage, entt::entity owner) {
    b2BodyDef bd;
    bd.type = b2_dynamicBody;
    bd.bullet = true;
    bd.position.Set(world_pos.x, world_pos.y);
    b2Body* body = worldPtr->CreateBody(&bd);
    b2CircleShape shape;
    shape.m_radius = 0.12f; // ~4 px
    b2FixtureDef fd;
    fd.shape = &shape;
    fd.density = 1.0f;
    fd.friction = 0.0f;
    fd.restitution = 0.0f;
    fd.isSensor = false;
    auto ent = registry.create();
    b2Fixture* fx = body->CreateFixture(&fd);
    // EDU - change to static_cast
    fx->GetUserData().pointer = static_cast<uintptr_t>(ent);
    registry.emplace<Body>(ent, Body{body});
    registry.emplace<Transform>(ent, Transform{ {world_pos.x, world_pos.y}, 0.0f });
    registry.emplace<Sprite>(ent, Sprite{ &gProjTex, 8, 8 });
    registry.emplace<Projectile>(ent, Projectile{ damage, static_cast<int>(owner), 2000.0 });
    registry.emplace<Lifetime>(ent, Lifetime{2000.0});
    // initial velocity
    float speed = 8.0f; // meters per second (~256 px/s)
    body->SetLinearVelocity(b2Vec2(dir.x * speed, dir.y * speed));
    return ent;
}

// --- Create the train entity and its Box2D body (kinematic body to be moved along track) ---
entt::entity create_train() {
    Vec2 pos = track_to_world(0.0f);
    b2BodyDef bd;
    bd.type = b2_kinematicBody;
    bd.position.Set(pos.x, pos.y);
    b2Body* body = worldPtr->CreateBody(&bd);
    b2CircleShape shape;
    shape.m_radius = 0.7f; // ~22 px
    b2FixtureDef fd;
    fd.shape = &shape;
    fd.density = 1.0f;
    fd.isSensor = false;
    auto ent = registry.create();
    b2Fixture* fx = body->CreateFixture(&fd);
    // EDU - change to static_cast
    fx->GetUserData().pointer = static_cast<uintptr_t>(ent);
    registry.emplace<Body>(ent, Body{body});
    registry.emplace<Transform>(ent, Transform{ {pos.x, pos.y}, 0.0f });
    registry.emplace<Sprite>(ent, Sprite{ &gTrainTex, 48, 48 });
    registry.emplace<Train>(ent, Train{ 0.0f, 2.5f, float(MAX_CARGO_HEALTH), 0.0, true, false });
    registry.emplace<Collider>(ent, Collider{22});
    return ent;
}

// --- Systems ---

// Sync transforms from Box2D to ENTT Transform
void sync_transforms_from_physics() {
    auto view = registry.view<Body, Transform>();
    for (auto ent : view) {
        auto &b = view.get<Body>(ent);
        if (!b.body) continue;
        b2Vec2 p = b.body->GetPosition();
        float a = b.body->GetAngle();
        auto &t = view.get<Transform>(ent);
        t.pos.x = p.x; t.pos.y = p.y; t.rot = a * 180.0f / b2_pi;
    }
}

// Render system - uses camera to convert world -> screen, train remains centered
void system_render() {
    SDL_SetRenderDrawColor(gRenderer, 24, 30, 40, 255);
    SDL_RenderClear(gRenderer);

    // draw repeating track texture in world coordinates around camera
    float tile_w_m = 4.0f; // meters per track tile width
    int start_i = int(std::floor((camera.center_world.x - (WINDOW_W*0.5f)*INV_PIXELS_PER_METER) / tile_w_m)) - 1;
    int end_i = int(std::ceil((camera.center_world.x + (WINDOW_W*0.5f)*INV_PIXELS_PER_METER) / tile_w_m)) + 1;
    for (int i = start_i; i <= end_i; ++i) {
        float wx = i * tile_w_m;
        float wy = std::sin((wx / TRACK_LENGTH) * 8.0f) * 3.5f;
        SDL_Point p = camera.world_to_screen(Vec2{ wx, wy });
        SDL_Rect dst{ p.x - (int)(tile_w_m * PIXELS_PER_METER / 2.0f), p.y - (int)(tile_w_m * PIXELS_PER_METER / 2.0f),
                      (int)std::round(tile_w_m * PIXELS_PER_METER), (int)std::round(tile_w_m * PIXELS_PER_METER) };
        SDL_RenderCopy(gRenderer, gTrackTex.tex, nullptr, &dst);
    }

    // render all entities with Sprite and Transform
    auto view = registry.view<Sprite, Transform>();
    for (auto ent : view) {
        auto &s = view.get<Sprite>(ent);
        auto &t = view.get<Transform>(ent);
        SDL_Point screen = camera.world_to_screen(t.pos);
        SDL_Rect dst{ screen.x - s.w/2, screen.y - s.h/2, s.w, s.h };
        SDL_RenderCopyEx(gRenderer, s.tex->tex, nullptr, &dst, t.rot, nullptr, SDL_FLIP_NONE);
    }

    // HUD (screen space)
    SDL_Color white{255,255,255,255};
    auto trainView = registry.view<Train>();
    for (auto ent : trainView) {
        auto &tr = trainView.get<Train>(ent);
        draw_text("Cargo HP: " + std::to_string((int)tr.cargo_health), 8, 8, white);
        draw_text("Time: " + std::to_string((int)tr.mission_clock) + "s / " + std::to_string((int)MISSION_TIME_SECONDS) + "s", 8, 28, white);
        draw_text("Speed (m/s): " + std::to_string(tr.speed_mps), 8, 48, white);
        draw_text("Controls: W/S fine speed +/-; Shift for larger step; Space to boost; 1-4 weapons; Click to fire", 8, 68, white);
    }

    // crosshair at mouse
    int mx, my;
    SDL_GetMouseState(&mx, &my);
    SDL_SetRenderDrawColor(gRenderer, 255, 220, 60, 200);
    SDL_RenderDrawLine(gRenderer, mx-10, my, mx+10, my);
    SDL_RenderDrawLine(gRenderer, mx, my-10, mx, my+10);

    SDL_RenderPresent(gRenderer);
}

// Step physics world
void physics_step(float dt) {
    const int32 velocityIter = 6;
    const int32 positionIter = 2;
    worldPtr->Step(dt, velocityIter, positionIter);
    sync_transforms_from_physics();
}

// Spawn monsters ahead of the train progress (meters)
uint32_t last_spawn_ms = 0;
void system_spawner() {
    uint32_t now = now_ms();
    if (now - last_spawn_ms < 700) return;
    last_spawn_ms = now;
    // spawn ahead of train progress
    auto v = registry.view<Train, Body>();
    for (auto ent : v) {
        auto &t = v.get<Train>(ent);
        float ahead = t.progress_m + 8.0f + std::uniform_real_distribution<float>(0.0f, 18.0f)(rng);
        if (ahead > TRACK_LENGTH) ahead = TRACK_LENGTH - 1.0f;
        create_monster(ahead);
    }
}

// Update train kinematic body along track according to target speed and boost; fine-step speed control applied in input handling
void system_update_train(float dt) {
    auto view = registry.view<Train, Body, Transform>();
    for (auto ent : view) {
        auto &train = view.get<Train>(ent);
        auto &b = view.get<Body>(ent);
        // if boost is on, temporarily increase speed
        float effective_speed = train.speed_mps + (train.boost ? 2.0f : 0.0f);
        // advance progress in meters
        train.progress_m += effective_speed * dt;
        if (train.progress_m > TRACK_LENGTH) train.progress_m = TRACK_LENGTH;
        // compute world pos for new progress and directly set kinematic body velocity so Box2D moves body to target pos
        Vec2 target = track_to_world(train.progress_m);
        b2Vec2 current = b.body->GetPosition();
        b2Vec2 desired(target.x, target.y);
        b2Vec2 vel = (desired - current);
        // EDU apply math to both components of this vecotr
        vel.x *= (1.0f / dt);
        vel.y *= (1.0f / dt);
        // clamp velocity to avoid huge numbers
        float vmax = 50.0f; // m/s clamp (safety)
        if (vel.Length() > vmax) vel *= (vmax / vel.Length());
        b.body->SetLinearVelocity(vel);
        // sync transform
        auto &tr = view.get<Transform>(ent);
        tr.pos = target;
        // update camera to center on train world pos
        camera.center_world = target;
        // update mission clock
        train.mission_clock += dt;
    }
}

// Weapon system: mouse click to fire; spawn projectile bodies
int selectedWeapon = 0;
bool firing = false;
void system_weapons_fire(int mx, int my, bool firingNow) {
    auto view = registry.view<Train, Transform>();
    for (auto ent : view) {
        auto &train = view.get<Train>(ent);
        auto &ttrans = view.get<Transform>(ent);
        static double lastFire = 0.0;
        double now = now_ms();
        // weapon presets (damage, cooldown_ms)
        struct W { int damage; int cooldown_ms; bool continuous; };
        static std::vector<W> weapons = { {8, 180, false}, {30, 600, false}, {5, 60, true}, {70, 1200, false} };
        int idx = std::min((int)weapons.size()-1, std::max(0, selectedWeapon));
        W w = weapons[idx];
        if (firingNow) {
            if (now - lastFire >= w.cooldown_ms) {
                // spawn projectile toward mouse cursor world pos
                Vec2 target = camera.screen_to_world(mx, my);
                Vec2 dir{ target.x - ttrans.pos.x, target.y - ttrans.pos.y };
                float len = std::sqrt(dir.x*dir.x + dir.y*dir.y);
                if (len > 0.001f) { dir.x /= len; dir.y /= len; }
                entt::entity owner = ent;
                create_projectile(ttrans.pos, dir, w.damage, owner);
                if (gFireSound) Mix_PlayChannel(-1, gFireSound, 0);
                lastFire = now;
            }
        }
    }
}

// Lifetime system removes entities whose lifetime expired and destroy their Box2D bodies
void system_lifetimes(double dt_ms) {
    std::vector<entt::entity> toDestroy;
    auto view = registry.view<Lifetime, Body>();
    for (auto ent : view) {
        auto &lt = view.get<Lifetime>(ent);
        lt.remain_ms -= dt_ms;
        if (lt.remain_ms <= 0.0) toDestroy.push_back(ent);
    }
    // Also remove monsters with hp <= 0
    auto mview = registry.view<Monster>();
    for (auto ent : mview) {
        auto &m = mview.get<Monster>(ent);
        if (m.hp <= 0) toDestroy.push_back(ent);
    }
    for (auto e : toDestroy) {
        if (!registry.valid(e)) continue;
        if (registry.all_of<Body>(e)) {
            auto &b = registry.get<Body>(e);
            if (b.body) {
                // destroy all fixtures then body
                worldPtr->DestroyBody(b.body);
                b.body = nullptr;
            }
        }
        registry.destroy(e);
    }
}

// Check mission end
bool check_mission_end(bool &success) {
    success = false;
    bool ended = false;
    auto view = registry.view<Train>();
    for (auto ent : view) {
        auto &t = view.get<Train>(ent);
        if (t.progress_m >= TRACK_LENGTH) {
            ended = true;
            if (t.mission_clock <= MISSION_TIME_SECONDS && t.cargo_health > 0) success = true;
            else success = false;
        } else if (t.mission_clock > MISSION_TIME_SECONDS) {
            ended = true; success = false;
        } else if (t.cargo_health <= 0) {
            ended = true; success = false;
        }
    }
    return ended;
}

// --- Main ---
int main(int argc, char** argv) {
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) != 0) {
        std::cerr << "SDL_Init failed: " << SDL_GetError() << "\n"; return 1;
    }
    if (TTF_Init() != 0) { std::cerr << "TTF_Init failed: " << TTF_GetError() << "\n"; return 1; }
    if (Mix_OpenAudio(44100, MIX_DEFAULT_FORMAT, 2, 2048) < 0) { std::cerr << "Mix_OpenAudio failed: " << Mix_GetError() << "\n"; }

    gWindow = SDL_CreateWindow("Train Defender (Box2D + ENTT)", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, WINDOW_W, WINDOW_H, SDL_WINDOW_SHOWN);
    gRenderer = SDL_CreateRenderer(gWindow, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);

    gFont = TTF_OpenFont(ASSET_FONT, 14);
    if (!gFont) std::cerr << "Failed to open font '" << ASSET_FONT << "'. Place a TTF font next to the executable.\n";

    if (!loadTextureBMP(ASSET_TRAIN, gTrainTex)) std::cerr << "Train texture missing or failed to load\n";
    if (!loadTextureBMP(ASSET_MONSTER, gMonsterTex)) std::cerr << "Monster texture missing or failed to load\n";
    if (!loadTextureBMP(ASSET_TRACK, gTrackTex)) std::cerr << "Track texture missing or failed to load\n";
    if (!loadTextureBMP(ASSET_PROJECTILE, gProjTex)) std::cerr << "Projectile texture missing or failed to load\n";
    if (!loadTextureBMP(ASSET_UI, gUiTex)) std::cerr << "UI texture missing or failed to load\n";

    gMusic = Mix_LoadMUS(ASSET_BG_MUSIC);
    gFireSound = Mix_LoadWAV(ASSET_FIRE_SOUND);
    gHitSound = Mix_LoadWAV(ASSET_HIT_SOUND);
    if (gMusic) Mix_PlayMusic(gMusic, -1);

    // create Box2D world with no gravity (top-down)
    worldPtr = std::make_unique<b2World>(b2Vec2(0.0f, 0.0f));
    ContactListener contactListener;
    worldPtr->SetContactListener(&contactListener);

    // create train
    entt::entity train_ent = create_train();

    // initialize camera center on train
    auto &trainTransform = registry.get<Transform>(train_ent);
    camera.center_world = trainTransform.pos;

    // game loop variables
    bool running = true;
    uint32_t last = now_ms();
    double accumulator_ms = 0.0;
    const double physics_dt_ms = 1000.0 / 60.0; // 60 Hz physics
    double accumulated_render = 0.0;

    // input state
    bool shiftDown = false;
    bool spaceDown = false;
    bool mouseDown = false;
    int mouseX = 0, mouseY = 0;

    // fine-step increments
    const float fineStep = 0.1f; // m/s per press
    const float coarseStep = 0.5f; // with Shift

    // main loop
    while (running) {
        uint32_t now = now_ms();
        double frame_ms = double(now - last);
        if (frame_ms < 0.0) frame_ms = 0.0;
        last = now;
        accumulator_ms += frame_ms;

        // event processing
        SDL_Event e;
        while (SDL_PollEvent(&e)) {
            if (e.type == SDL_QUIT) running = false;
            else if (e.type == SDL_MOUSEBUTTONDOWN) {
                if (e.button.button == SDL_BUTTON_LEFT) mouseDown = true;
            } else if (e.type == SDL_MOUSEBUTTONUP) {
                if (e.button.button == SDL_BUTTON_LEFT) mouseDown = false;
            } else if (e.type == SDL_MOUSEMOTION) {
                mouseX = e.motion.x; mouseY = e.motion.y;
            } else if (e.type == SDL_KEYDOWN) {
                if (e.key.keysym.sym == SDLK_ESCAPE) running = false;
                else if (e.key.keysym.sym == SDLK_LSHIFT || e.key.keysym.sym == SDLK_RSHIFT) shiftDown = true;
                else if (e.key.keysym.sym == SDLK_SPACE) {
                    // toggle boost while space held
                    spaceDown = true;
                    auto view = registry.view<Train>();
                    for (auto entt : view) view.get<Train>(entt).boost = true;
                } else if (e.key.keysym.sym == SDLK_w || e.key.keysym.sym == SDLK_UP) {
                    // increase speed
                    auto v = registry.view<Train>();
                    for (auto entt : v) {
                        float step = shiftDown ? coarseStep : fineStep;
                        v.get<Train>(entt).speed_mps += step;
                    }
                } else if (e.key.keysym.sym == SDLK_s || e.key.keysym.sym == SDLK_DOWN) {
                    auto v = registry.view<Train>();
                    for (auto entt : v) {
                        float step = shiftDown ? coarseStep : fineStep;
                        v.get<Train>(entt).speed_mps -= step;
                        if (v.get<Train>(entt).speed_mps < 0.0f) v.get<Train>(entt).speed_mps = 0.0f;
                    }
                } else if (e.key.keysym.sym >= SDLK_1 && e.key.keysym.sym <= SDLK_4) {
                    selectedWeapon = e.key.keysym.sym - SDLK_1;
                }
            } else if (e.type == SDL_KEYUP) {
                if (e.key.keysym.sym == SDLK_LSHIFT || e.key.keysym.sym == SDLK_RSHIFT) shiftDown = false;
                else if (e.key.keysym.sym == SDLK_SPACE) {
                    spaceDown = false;
                    auto view = registry.view<Train>();
                    for (auto entt : view) view.get<Train>(entt).boost = false;
                }
            }
        }

        // physics fixed-step loop
        while (accumulator_ms >= physics_dt_ms) {
            float dt_sec = (float)(physics_dt_ms * 0.001);
            // spawn monsters periodically
            system_spawner();

            // update train kinematic body and progress
            system_update_train(dt_sec);

            // fire weapons if mouseDown
            SDL_GetMouseState(&mouseX, &mouseY);
            if (mouseDown) system_weapons_fire(mouseX, mouseY, true);

            // advance physics
            physics_step(dt_sec);

            // lifetimes in ms
            system_lifetimes(physics_dt_ms);

            accumulator_ms -= physics_dt_ms;
        }

        // render once per frame (not necessarily synced to physics loop)
        // sync transforms from physics bodies (already done after physics_step), but do again to be safe
        sync_transforms_from_physics();

        system_render();

        // check mission end
        bool success = false;
        if (check_mission_end(success)) {
            SDL_SetRenderDrawColor(gRenderer, 0, 0, 0, 220);
            SDL_Rect r{0,0,WINDOW_W, WINDOW_H};
            SDL_RenderFillRect(gRenderer, &r);
            SDL_Color col{255,255,255,255};
            if (success) draw_text("MISSION SUCCESS: Cargo delivered and on time", 140, WINDOW_H/2 - 10, col);
            else draw_text("MISSION FAILED: Train late or cargo destroyed", 140, WINDOW_H/2 - 10, col);
            SDL_RenderPresent(gRenderer);
            SDL_Delay(3500);
            running = false;
        }

        SDL_Delay(5);
    }

    // cleanup: destroy remaining Box2D bodies gracefully
    // destroy bodies via world
    worldPtr.reset();

    if (gMusic) Mix_FreeMusic(gMusic);
    if (gFireSound) Mix_FreeChunk(gFireSound);
    if (gHitSound) Mix_FreeChunk(gHitSound);
    if (gFont) TTF_CloseFont(gFont);
    if (gTrainTex.tex) SDL_DestroyTexture(gTrainTex.tex);
    if (gMonsterTex.tex) SDL_DestroyTexture(gMonsterTex.tex);
    if (gTrackTex.tex) SDL_DestroyTexture(gTrackTex.tex);
    if (gProjTex.tex) SDL_DestroyTexture(gProjTex.tex);
    if (gUiTex.tex) SDL_DestroyTexture(gUiTex.tex);
    if (gRenderer) SDL_DestroyRenderer(gRenderer);
    if (gWindow) SDL_DestroyWindow(gWindow);
    Mix_CloseAudio();
    TTF_Quit();
    SDL_Quit();
    return 0;
}
